<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Mail;

class SettingController extends Controller
{
    /**
     * Menampilkan halaman pengaturan.
     */
    public function index()
    {
        $path = storage_path('app/settings.json');
        $settings = [];

        if (File::exists($path)) {
            $settings = json_decode(File::get($path), true);
        }

        return view('setting.index', compact('settings'));
    }

    /**
     * Mengupdate dan menyimpan pengaturan.
     */
    public function update(Request $request)
    {
        // [DITAMBAHKAN] Validasi untuk field SMTP baru
        $request->validate([
            'company_name' => 'required|string|max:255',
            'company_address' => 'nullable|string',
            'company_phone' => 'nullable|string',
            'company_logo' => 'nullable|image|mimes:jpeg,png,jpg|max:1024', // max 1MB

            // Aturan validasi untuk SMTP
            'mail_mailer' => 'nullable|string|max:100',
            'mail_host' => 'nullable|string|max:255',
            'mail_port' => 'nullable|string|max:10',
            'mail_username' => 'nullable|string|max:255',
            'mail_password' => 'nullable|string|max:255', // Dibuat nullable agar bisa "biarkan kosong"
            'mail_encryption' => 'nullable|string|max:10',
            'mail_from_address' => 'nullable|email|max:255',
            'mail_from_name' => 'nullable|string|max:255',
        ]);
        
        // INI ADALAH LANJUTAN FUNGSI UPDATE
        $path = storage_path('app/settings.json');
        
        // Ambil pengaturan lama jika file sudah ada
        $settings = File::exists($path) ? json_decode(File::get($path), true) : [];

        // Update data umum
        $settings['company_name'] = $request->input('company_name');
        $settings['company_address'] = $request->input('company_address');
        $settings['company_phone'] = $request->input('company_phone');

        // [DITAMBAHKAN] Update data SMTP
        $settings['mail_mailer'] = $request->input('mail_mailer');
        $settings['mail_host'] = $request->input('mail_host');
        $settings['mail_port'] = $request->input('mail_port');
        $settings['mail_username'] = $request->input('mail_username');
        $settings['mail_encryption'] = $request->input('mail_encryption');
        $settings['mail_from_address'] = $request->input('mail_from_address');
        $settings['mail_from_name'] = $request->input('mail_from_name');

        // [DIUBAH] Logika khusus untuk password:
        // Hanya update password jika pengguna mengisinya (tidak kosong).
        // Jika kosong, biarkan password lama tetap tersimpan.
        if ($request->filled('mail_password')) {
            $settings['mail_password'] = $request->input('mail_password');
        }

        // Proses upload logo jika ada
        if ($request->hasFile('company_logo')) {
            // Hapus logo lama jika ada
            if (isset($settings['company_logo'])) {
                Storage::disk('public')->delete($settings['company_logo']);
            }
            
            // Simpan logo baru di storage/app/public/logos
            $logoPath = $request->file('company_logo')->store('logos', 'public');
            $settings['company_logo'] = $logoPath;
        }

        // Simpan kembali ke file JSON
        File::put($path, json_encode($settings, JSON_PRETTY_PRINT));

        return redirect()->route('settings.index')->with('success', 'Pengaturan berhasil diperbarui.');
    } // <-- INI ADALAH PENUTUP FUNGSI UPDATE


    /**
     * [FUNGSI BARU]
     * Mencoba mengirim email tes menggunakan pengaturan yang tersimpan.
     * * FUNGSI INI SEHARUSNYA BERADA DI LUAR FUNGSI UPDATE, TAPI DI DALAM CLASS
     */
    public function testSmtp(Request $request)
    {
        $request->validate(['test_email' => 'required|email']);
        $testEmail = $request->input('test_email');

        // 1. Ambil pengaturan dari file JSON
        $path = storage_path('app/settings.json');
        if (!File::exists($path)) {
            return redirect()->back()->with('test_error', 'Pengaturan belum disimpan. Harap simpan pengaturan terlebih dahulu.');
        }
        $settings = json_decode(File::get($path), true);

        // 2. Set konfigurasi email Laravel secara dinamis
        try {
            Config::set('mail.default', 'smtp');
            Config::set('mail.mailers.smtp.host', $settings['mail_host'] ?? null);
            Config::set('mail.mailers.smtp.port', $settings['mail_port'] ?? null);
            Config::set('mail.mailers.smtp.encryption', $settings['mail_encryption'] ?? null);
            Config::set('mail.mailers.smtp.username', $settings['mail_username'] ?? null);
            Config::set('mail.mailers.smtp.password', $settings['mail_password'] ?? null);
            Config::set('mail.from.address', $settings['mail_from_address'] ?? 'noreply@example.com');
            Config::set('mail.from.name', $settings['mail_from_name'] ?? 'Aplikasi Anda');

            // 3. Coba kirim email
            Mail::raw('Ini adalah email tes untuk memverifikasi konfigurasi SMTP Anda.', function ($message) use ($testEmail) {
                $message->to($testEmail)
                        ->subject('Tes Koneksi SMTP Berhasil');
            });

            // 4. Handle Sukses
            return redirect()->back()->with('test_success', 'Email tes berhasil dikirim ke ' . $testEmail);

        } catch (\Exception $e) {
            // 5. Handle Gagal
            return redirect()->back()->with('test_error', 'Gagal: ' . $e->getMessage());
        }
    }
} // <-- INI ADALAH PENUTUP CLASS CONTROLLER